﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using FyndSharp.Utilities.Threading;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.VBECS.Communication.Clients;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Protocols;

namespace gov.va.med.VBECS.Communication.Channels
{
    /// <summary>
    /// This class adds SendMessageAndWaitForResponse(...) and SendAndReceiveMessage methods
    /// to a IMessenger for synchronous request/response style messaging.
    /// It also adds queued processing of incoming messages.
    /// </summary>
    /// <typeparam name="T">Type of IMessenger object to use as underlying communication</typeparam>
    /// <typeparam name="TP">Type of IPinger to use as ping functionality</typeparam>
    public class SequentialMessager<T, TP> : ISequentialMessager, IDisposable
        where T : IMessager
        where TP : IPinger, new()
    {
        #region Public events

        /// <summary>
        /// This event is raised when a new message is received from underlying messenger.
        /// </summary>
        public event EventHandler<MessageEventArgs> MessageReceived;

        /// <summary>
        /// This event is raised when a new message is sent without any error.
        /// It does not guaranties that message is properly handled and processed by remote application.
        /// </summary>
        public event EventHandler<MessageEventArgs> MessageSent;

        #endregion

        #region Public properties

        /// <summary>
        /// Gets the underlying IClient object.
        /// </summary>
        public T Messager { get; private set; }

        /// <summary>
        /// Gets the underlying IClient object.
        /// </summary>
        public IClient UnderlyingMessager
        {
            get { return Messager as IClient; }
        }

        /// <summary>
        /// Timeout value as milliseconds to wait for a receiving message on 
        /// SendMessageAndWaitForResponse and SendAndReceiveMessage methods.
        /// Default value: 60000 (1 minute).
        /// </summary>
        public int Timeout { get; set; }

        /// <summary>
        /// Gets/sets wire protocol that is used while reading and writing messages.
        /// </summary>
        public IProtocol Protocol
        {
            get { return Messager.Protocol; }
            set { Messager.Protocol = value; }
        }

        /// <summary>
        /// Gets the time of the last successfully received message.
        /// </summary>
        public DateTime LastReceivedTime
        {
            get { return Messager.LastReceivedTime; }
        }

        /// <summary>
        /// Gets the time of the last successfully received message.
        /// </summary>
        public DateTime LastSentTime
        {
            get { return Messager.LastSentTime; }
        }

        /// <summary>
        /// Indicates if the messager is currently busy
        /// </summary>
        public bool IsBusy
        {
            get
            {
                return _waitingMessages.Count > 0;
            }
        }

        #endregion

        #region Private fields

        /// <summary>
        /// ManualResetEvent to block thread until response is received.
        /// </summary>
        private readonly ManualResetEvent _waitEvent;

        /// <summary>
        /// This object is used to process incoming messages sequentially.
        /// </summary>
        private readonly SequentialItemProcessor<IMessage> _incomingMessageProcessor;

        /// <summary>
        /// This object is used for thread synchronization.
        /// </summary>
        private readonly Object _syncObject = new Object();

        /// <summary>
        /// This messages are waiting for a response those are used when 
        /// send_message_and_wait_for_response is called.
        /// Value: A WaitingMessage instance.
        /// </summary>
        private readonly List< WaitingMessage> _waitingMessages;

        /// <summary>
        /// Pinger object
        /// </summary>
        private readonly TP _pinger;

        // Logger object
        readonly ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        #endregion

        #region Constructor

        /// <summary>
        /// Creates a new RequestReplyMessenger.
        /// </summary>
        /// <param name="theMessager">IMessenger object to use as underlying communication</param>
        public SequentialMessager(T theMessager)
        {
            Messager = theMessager;
            _incomingMessageProcessor = new SequentialItemProcessor<IMessage>(send_message_and_wait_for_response);
            //_resultsMessageProcessor = new SequentialItemProcessor<WaitingMessage>(wait_message_processed_event);
            _waitingMessages = new List<WaitingMessage>();
            _waitEvent = new ManualResetEvent(false);
            // create pinger object.
            _pinger = new TP();
        }

        #endregion

        #region Public methods

        #region IDisposable Members

        /// <summary>
        /// Calls Stop method of this object.
        /// </summary>
        public void Dispose()
        {
            Stop();
        }

        #endregion

        #region IMessager Members

        /// <summary>
        /// Sends a message.
        /// </summary>
        /// <param name="aMessage">Message to be sent</param>
        public void Send(IMessage aMessage)
        {
            Messager.Send(aMessage);
        }

        #endregion

        /// <summary>
        /// Starts the messenger.
        /// </summary>
        public virtual void Start()
        {
            _logger.Debug("Start");
            _incomingMessageProcessor.Start();
            Messager.MessageReceived += messager_message_received;
            Messager.MessageSent += messager_message_sent;
            // start pinger object
            _pinger.Start(this);
        }

        /// <summary>
        /// Stops the messenger.
        /// Cancels all waiting threads in SendMessageAndWaitForResponse method and stops message queue.
        /// SendMessageAndWaitForResponse method throws exception if there is a thread that is waiting for response message.
        /// Also stops incoming message processing and deletes all messages in incoming message queue.
        /// </summary>
        public virtual void Stop()
        {
            _logger.Debug("Stop");
            // before processor stops it wait when current operation finishes.
            _incomingMessageProcessor.Stop();
            Messager.MessageReceived -= messager_message_received;
            Messager.MessageSent -= messager_message_sent;

            //Pulse waiting threads for incoming messages, since underlying messenger is disconnected
            //and can not receive messages anymore.
            lock (_syncObject)
            {
                _waitEvent.Set();
                foreach (var waitingMessage in _waitingMessages)
                {
                    waitingMessage.Status = WaitingMessageStatus.Cancelled;
                    waitingMessage.WaitEvent.Set();
                }

                _waitingMessages.Clear();
            }
            _logger.Debug("Stopped");
        }

        /// <summary>
        /// Sends a message.
        /// </summary>
        /// <param name="aMessage">Message to be sent</param>
        /// <param name="theTimeoutMilliseconds">Timeout duration as milliseconds.</param>
        /// <param name="messageReseivedMethod">Method is called when reply message arrive</param>
        public void Send(IMessage aMessage, int theTimeoutMilliseconds, Action<IMessage> messageReseivedMethod)
        {
            if (UnderlyingMessager.CommunicationStatus != CommunicationStatus.Connected)
            {
                _logger.Error("Send: Client is not connected to the server.");
                throw new CommunicationException("Client is not connected to the server.");
            }
            //Create a waiting message record and add to list
            var waitingMessage = new WaitingMessage(theTimeoutMilliseconds, messageReseivedMethod);
            lock (_syncObject)
            {
                _waitingMessages.Add(waitingMessage);
            }

            //Enqueue message for processing
            _incomingMessageProcessor.EnqueueMessage(aMessage);
        }

        /// <summary>
        /// Sends and Receives a message.
        /// </summary>
        /// <param name="aMessage">Message to be sent</param>
        /// <param name="theTimeoutMilliseconds">Timeout duration as milliseconds.</param>
        public IMessage SendReceive(IMessage aMessage, int theTimeoutMilliseconds)
        {
            var wEvent = new ManualResetEvent(false);
            IMessage returnMessage = null;
            Send(aMessage, theTimeoutMilliseconds,
                                            delegate(IMessage message)
                                                {
                                                    returnMessage = message;
                                                    wEvent.Set();
                                                });

            wEvent.WaitOne(-1);
            // check is there any errors (GetBytes() will throw an exception is any errors)
            //CR 3554
            if (returnMessage != null)
            {
                byte[] bytes = returnMessage.GetBytes();
                if (bytes == null)
                    throw new ApplicationException("bytes is null");
            }
            return returnMessage;
        }

        #endregion

        #region Private methods

        /// <summary>
        /// Sends a message and waits a response for that message.
        /// </summary>
        /// <remarks>
        /// Response message is matched by order. Next received message after send is considered as a reply.
        /// 
        /// MessageReceived event is not raised for response messages.
        /// </remarks>
        /// <param name="aMessage">message to send</param>
        /// <exception cref="TimeoutException">Throws TimeoutException if can not receive reply message in timeout value</exception>
        /// <exception cref="CommunicationException">Throws CommunicationException if communication fails before reply message.</exception>
        private void send_message_and_wait_for_response(IMessage aMessage)
        {
            //Locate waiting message. This is always first message in the waiting list.
            
            // TODO: test this exceptions
            if (string.IsNullOrEmpty(aMessage.Id))
            {
                _logger.Error("send_message_and_wait_for_response: Message without ID was enqueued.");
                throw new ApplicationException("Message without ID was enqueued");
            }

            WaitingMessage waitingMessage = null;
            lock (_syncObject)
            {
                if (_waitingMessages.Count > 0)
                {
                    waitingMessage = _waitingMessages[0];
                    if (waitingMessage != null)
                    {
                        waitingMessage.Status = WaitingMessageStatus.WaitingForResponse;
                        //Pulse messages' wait event and start listen it here.
                        waitingMessage.WaitEvent.Set();
                    }
                }
            }

            if (waitingMessage == null)
            {
                _logger.Error("send_message_and_wait_for_response: Enqueued message was not found.");
                throw new ApplicationException("Enqueued message was not found");
            }

            //Send and wait for the reply
            try
            {
                //Send message
                Messager.Send(aMessage);

                //Wait for response
                _waitEvent.WaitOne(waitingMessage.TheTimeoutMilliseconds);

                //Check for exceptions
                switch (waitingMessage.Status)
                {
                    case WaitingMessageStatus.WaitingForResponse:
                    {
                        _logger.Error(
                            string.Format(
                                "send_message_and_wait_for_response: Timeout occurred. Can not receive response.{0}Message: {1}",
                                Environment.NewLine, Encoding.UTF8.GetString(aMessage.GetBytes())));
                        waitingMessage.MessageReseivedMethod(
                            new ErrorMessage(new TimeoutException("Timeout occurred. Can not receive response.")));
                        return;
                    }
                    case WaitingMessageStatus.Cancelled:
                    {
                        _logger.Error(
                            string.Format(
                                "send_message_and_wait_for_response: Disconnected before response received.{0}Message: {1}",
                                Environment.NewLine, Encoding.UTF8.GetString(aMessage.GetBytes())));
                        waitingMessage.MessageReseivedMethod(
                            new ErrorMessage(new CommunicationException("Disconnected before response received.")));
                        return;
                    }
                }

                //fire response method
                waitingMessage.MessageReseivedMethod(waitingMessage.ResponseMessage);
            }
            catch (Exception ex)
            {
                _logger.Error(
                            string.Format(
                                "send_message_and_wait_for_response: Exception received.{0}Message: {1}{0}Error: {2}",
                                Environment.NewLine, Encoding.UTF8.GetString(aMessage.GetBytes()), ex.Message));
                waitingMessage.MessageReseivedMethod(new ErrorMessage(ex));
            }
            finally
            {
                //Remove message from waiting messages
                lock (_syncObject)
                {
                    if (_waitingMessages.Count > 0)
                    {
                        _waitingMessages.RemoveAt(0);
                    }

                    _waitEvent.Reset();
                }
            }
        }

        /// <summary>
        /// Handles MessageReceived event of Messenger object.
        /// </summary>
        /// <param name="sender">Source of event</param>
        /// <param name="e">Event arguments</param>
        private void messager_message_received(object sender, MessageEventArgs e)
        {
            //There is always one thread waiting thread for this message in send_message_and_wait_for_response method
            WaitingMessage waitingMessage = null;
            lock (_syncObject)
            {
                if (_waitingMessages.Count > 0)
                {
                    waitingMessage = _waitingMessages[0];
                }
            }

            if (waitingMessage != null)
            {
                //throw new ApplicationException("Enqueued message was not found");

                //Enqueued message was not found, just fire message received event
                //TODO: assign appropriate ID here
                //e.Message.RepliedId = waitingMessage.ID

                waitingMessage.ResponseMessage = e.Message;
                waitingMessage.Status = WaitingMessageStatus.ResponseReceived;
                _waitEvent.Set();
            }


            FireMessageReceivedEvent(e.Message);
        }

        /// <summary>
        /// Handles MessageSent event of Messenger object.
        /// </summary>
        /// <param name="sender">Source of event</param>
        /// <param name="e">Event arguments</param>
        private void messager_message_sent(object sender, MessageEventArgs e)
        {
            FireMessageSentEvent(e.Message);
        }

        #endregion

        #region Event raising methods

        /// <summary>
        /// Raises MessageReceived event.
        /// </summary>
        /// <param name="theMessage">Received message</param>
        protected virtual void FireMessageReceivedEvent(IMessage theMessage)
        {
            var handler = MessageReceived;
            if (handler != null)
            {
                handler(this, new MessageEventArgs(theMessage));
            }
        }

        /// <summary>
        /// Raises MessageSent event.
        /// </summary>
        /// <param name="theMessage">Received message</param>
        protected virtual void FireMessageSentEvent(IMessage theMessage)
        {
            var handler = MessageSent;
            if (handler != null)
            {
                handler(this, new MessageEventArgs(theMessage));
            }
        }

        #endregion

        #region WaitingMessage class

        #region Nested type: WaitingMessage

        /// <summary>
        /// This class is used to store messaging context for a request message
        /// until response is received.
        /// </summary>
// ReSharper disable InconsistentNaming
        private sealed class WaitingMessage
// ReSharper restore InconsistentNaming
        {
            /// <summary>
            /// Creates a new WaitingMessage object.
            /// </summary>
            public WaitingMessage(int theTimeoutMilliseconds, Action<IMessage> messageReseivedMethod)
            {
                WaitEvent = new ManualResetEvent(false);
                Status = WaitingMessageStatus.Queued;
                MessageReseivedMethod = messageReseivedMethod;
                TheTimeoutMilliseconds = theTimeoutMilliseconds;

                // Wait process in the separate thread
                // Task.Run(() => do_start(_tokenSource.Token), _tokenSource.Token);
                // Check here for aggregation exception.
                Task.Factory.StartNew(do_wait).ContinueWith(
                        t => MessageReseivedMethod(new ErrorMessage(t.Exception == null ? null : t.Exception.InnerException)), 
                        TaskContinuationOptions.OnlyOnFaulted);
            }

            /// <summary>
            /// Response message for request message 
            /// (null if response is not received yet).
            /// </summary>
            public IMessage ResponseMessage { get; set; }

            /// <summary>
            /// ManualResetEvent to block thread until message processing started.
            /// </summary>
            public ManualResetEvent WaitEvent { get; private set; }

            /// <summary>
            /// State of the request message.
            /// </summary>
            public WaitingMessageStatus Status { get; set; }

            /// <summary>
            /// The method delegate that is called to actually process items.
            /// </summary>
            public Action<IMessage> MessageReseivedMethod { get; private set; }


            /// <summary>
            /// Timeout duration as milliseconds.
            /// </summary>
            public int TheTimeoutMilliseconds { get; private set; }

            /// <summary>
            /// wait method
            /// </summary>
            private void do_wait()
            {
                //Wait for response
                WaitEvent.WaitOne(-1);

                //Check for exceptions
                switch (Status)
                {
                    case WaitingMessageStatus.WaitingForResponse:
                        {
                            return; // message transmission started. Stop listening it here.
                        }
                    case WaitingMessageStatus.Cancelled:
                        throw new CommunicationException("Disconnected before request started.");
                }

            }

        }

        #endregion

        #region Nested type: WaitingMessageStatus

        /// <summary>
        /// This enum is used to store the state of a waiting message.
        /// </summary>
        private enum WaitingMessageStatus
        {
            /// <summary>
            /// Queued for processing.
            /// </summary>
            Queued,

            /// <summary>
            /// Still waiting for response.
            /// </summary>
            WaitingForResponse,

            /// <summary>
            /// Message sending is cancelled.
            /// </summary>
            Cancelled,

            /// <summary>
            /// Response is properly received.
            /// </summary>
            ResponseReceived
        }

        #endregion

        #endregion
    }
}